/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | www.openfoam.com
     \\/     M anipulation  |
-------------------------------------------------------------------------------
    Copyright (C) 2011-2017 OpenFOAM Foundation
    Copyright (C) 2015-2018 OpenCFD Ltd.
-------------------------------------------------------------------------------
License
    This file is part of OpenFOAM.

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "spaceTimeVaryingMappedFixedValueFvPatchField.H"
#include "Time.H"

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class Type>
Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::
spaceTimeVaryingMappedFixedValueFvPatchField
(
    const fvPatch& p,
    const DimensionedField<Type, volMesh>& iF
)
:
    fixedValueFvPatchField<Type>(p, iF),
    spacePattern_(),
    timePattern_(nullptr),
    scaling_()
{}


template<class Type>
Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::
spaceTimeVaryingMappedFixedValueFvPatchField
(
    const fvPatch& p,
    const DimensionedField<Type, volMesh>& iF,
    const dictionary& dict
)
:
    fixedValueFvPatchField<Type>(p, iF, dict, IOobjectOption::NO_READ),
    spacePattern_
    (
        new PatchFunction1Types::MappedFile<Type>
        (
            p.patch(),
            "spacePattern",
            dict,
            iF.name(),          // field table name
            true                // face values
        )
    ),
    timePattern_
    (
        Function1<scalar>::New("timePattern", dict)
    ),
    scaling_(dict.getOrDefault<scalar>("scaling", 1.0))
{
    if (!this->readValueEntry(dict))
    {
        // Note: we use evaluate() here to trigger updateCoeffs followed
        //       by re-setting of fvatchfield::updated_ flag. This is
        //       so if first use is in the next time step it retriggers
        //       a new update.
        this->evaluate(Pstream::commsTypes::buffered);
    }
}


template<class Type>
Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::
spaceTimeVaryingMappedFixedValueFvPatchField
(
    const spaceTimeVaryingMappedFixedValueFvPatchField<Type>& ptf,
    const fvPatch& p,
    const DimensionedField<Type, volMesh>& iF,
    const fvPatchFieldMapper& mapper
)
:
    fixedValueFvPatchField<Type>(ptf, p, iF, mapper),
    spacePattern_
    (
        new PatchFunction1Types::MappedFile<Type>
        (
            ptf.spacePattern_(),
            p.patch()
        )
    ),
    timePattern_(ptf.timePattern_.clone()),
    scaling_(ptf.scaling_)
{}


template<class Type>
Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::
spaceTimeVaryingMappedFixedValueFvPatchField
(
    const spaceTimeVaryingMappedFixedValueFvPatchField<Type>& ptf
)
:
    fixedValueFvPatchField<Type>(ptf),
    spacePattern_
    (
        new PatchFunction1Types::MappedFile<Type>
        (
            ptf.spacePattern_(),
            this->patch().patch()
        )
    ),
    timePattern_(ptf.timePattern_.clone()),
    scaling_(ptf.scaling_)
{}


template<class Type>
Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::
spaceTimeVaryingMappedFixedValueFvPatchField
(
    const spaceTimeVaryingMappedFixedValueFvPatchField<Type>& ptf,
    const DimensionedField<Type, volMesh>& iF
)
:
    fixedValueFvPatchField<Type>(ptf, iF),
    spacePattern_
    (
        new PatchFunction1Types::MappedFile<Type>
        (
            ptf.spacePattern_(),
            this->patch().patch()
        )
    ),
    timePattern_(ptf.timePattern_.clone()),
    scaling_(ptf.scaling_)
{}


// * * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * //

template<class Type>
void Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::autoMap
(
    const fvPatchFieldMapper& m
)
{
    fixedValueFvPatchField<Type>::autoMap(m);
    spacePattern_().autoMap(m);
}


template<class Type>
void Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::rmap
(
    const fvPatchField<Type>& ptf,
    const labelList& addr
)
{
    fixedValueFvPatchField<Type>::rmap(ptf, addr);

    const spaceTimeVaryingMappedFixedValueFvPatchField<Type>& tiptf =
        refCast<const spaceTimeVaryingMappedFixedValueFvPatchField<Type>>(ptf);

    spacePattern_().rmap(tiptf.spacePattern_(), addr);
}


template<class Type>
void Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::updateCoeffs()
{
    if (this->updated())
    {
        return;
    }

    const scalar t = this->db().time().value();

    fvPatchField<Type>::operator==(spacePattern_->value(t)
	                           *timePattern_->value(t)
	                           *scaling_);

    fixedValueFvPatchField<Type>::updateCoeffs();
}


template<class Type>
void Foam::spaceTimeVaryingMappedFixedValueFvPatchField<Type>::write
(
    Ostream& os
) const
{
    fvPatchField<Type>::write(os);
    spacePattern_->writeData(os);
    timePattern_->writeData(os);
    os.writeEntry("scaling", scaling_);
    fvPatchField<Type>::writeValueEntry(os);
}


// ************************************************************************* //
